/* Copyright (C) 2000  The PARI group.

This file is part of the PARI/GP package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later
version. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA. */
#include "pari.h"
#include "paripriv.h"

#define DEBUGLEVEL DEBUGLEVEL_mathnf

#define dbg_printf(lvl) if (DEBUGLEVEL >= (lvl) + 3) err_printf

/********************************************************************/
/**                                                                **/
/**                      AMT TODO BETTER TITLE                     **/
/**                 contributed by Aurel Page (2021)               **/
/**             after Leon Lampret TODO paper citation             **/
/**                                                                **/
/********************************************************************/

/*
 * A KzMs (complex of sparse matrices) is a t_VEC with two components:
 * - a t_VECSMALL of dimensions d_0,...,d_n
 * - a t_VEC of zMs D_1,...,D_n representing differentials
 *   D_i : Z^{d_{i-1}} <- Z^{d_i}
 *   s.t. D_i*D_{i+1} = 0 for all 1<=i<n.
 */

static int
isinv(long n)
{
  return (n==1) || (n==-1);
}

static GEN
steepness_matching(GEN dims, GEN diffs)
{
  GEN low, M, W, D, E, lowk, C, Mk, Wk, zero_zCs, Mk1;
  long k, n, v, i, u, w, i2;
  n = lg(diffs)-1;

  zero_zCs = mkcol2(cgetg(1,t_VECSMALL),cgetg(1,t_VECSMALL));

  low = cgetg(n+1, t_VEC);
  W = cgetg(n+1, t_VEC);
  for (k = 1; k <= n; k++)
  {
    gel(low,k) = zero_zv(dims[k]);
    gel(W,k) = zero_zv(dims[k]);
  }
  M = cgetg(n+2, t_VEC);
  for (k = 1; k <= n+1; k++) gel(M,k) = zero_zv(dims[k]);

  /* first pass: lowest incoming edge */
  for (k = 1; k <= n; k++)
  {
    D = gel(diffs,k);
    lowk = gel(low,k);
    for (v = dims[k+1]; v > 0; v--)
    {
      C = gmael(D,v,1);
      for (i = 1; i < lg(C); i++) lowk[C[i]] = v;
    }
  }

  /* second pass: compute matching */
  for (k = 1; k <= n; k++)
  {
    D = gel(diffs,k);
    lowk = gel(low,k);
    Mk = gel(M,k);
    Mk1 = gel(M,k+1);
    Wk = gel(W,k);
    for (v = 1; v <= dims[k+1]; v++)
    {
      C = gmael(D,v,1);
      if (lg(C)>1)
      {
        E = gmael(D,v,2);
        u = C[lg(E)-1];
        w = E[lg(E)-1];
        if (lowk[u]==v && isinv(w))
        {
          /* match (u,v) */
          Mk[u] = v;
          Wk[u] = -w; /* == -1/w */
          Mk1[v] = -1;
          if (k > 1) gmael(diffs,k-1,u) = gcopy(zero_zCs);
        }
      }
    }
  }

  /* third pass: cleanup */
  for (k = 1; k <= n; k++)
  {
    D = gel(diffs,k);
    Mk = gel(M,k);
    Wk = gel(W,k);
    for (v = 1; v <= dims[k+1]; v++)
    {
      C = gmael(D,v,1);
      E = gmael(D,v,2);
      i2 = 1;
      for (i=1; i<lg(C); i++)
      {
        u = C[i];
        if (Mk[u] != -1 && Mk[u] != v) /* else delete entry */
        {
          if (Mk[u] > 0) E[i] *= Wk[u];
          if (i2 < i)
          {
            C[i2] = C[i];
            E[i2] = E[i];
          }
          i2++;
        }
      }
      setlg(C,i2);
      setlg(E,i2);
    }
  }

  M = mkvec2(M,W);
  return M;
}

static GEN
mergeres(GEN C0, GEN E0)
{
  GEN perm, C, E;
  long i, j;
  if (lg(C0) == 1) return mkvec2(cgetg(1,t_VECSMALL),cgetg(1,t_VECSMALL));
  C0 = shallowconcat1(C0);
  E0 = shallowconcat1(E0);
  perm = vecsmall_indexsort(C0);
  C0 = vecsmallpermute(C0,perm);
  E0 = vecsmallpermute(E0,perm);
  C = cgetg(lg(C0),t_VECSMALL);
  E = cgetg(lg(C0),t_VECSMALL);
  for (i = 1, j = 0; i < lg(C); i++)
  {
    if (j>0 && C[j]==C0[i]) E[j] += E0[i];
    else
    {
      j++;
      C[j] = C0[i];
      E[j] = E0[i];
    }
  }
  setlg(C,j+1);
  setlg(E,j+1);
  for (i = 1, j = 0; i < lg(C); i++)
  {
    if (E[i])
    {
      j++;
      if (i>j)
      {
        C[j] = C[i];
        E[j] = E[i];
      }
    }
  }
  setlg(C,j+1);
  setlg(E,j+1);
  return mkcol2(C, E);
}

/* no GC */
/* at the moment, target == unmatched TODO more general */
static GEN
zigzag(long v, GEN M, GEN diff)
{
  GEN C, E, resC, resE, c;
  long i, u, w;
  C = gmael(diff,v,1);
  E = gmael(diff,v,2);
  resC = cgetg(lg(C),t_COL);
  resE = cgetg(lg(E),t_COL);
  for (i = 1; i<lg(C); i++)
  {
    u = C[i];
    w = E[i];
    if (!M[u])
    {
      gel(resC,i) = mkvecsmall(u);
      gel(resE,i) = mkvecsmall(w);
    }
    else
    {
      c = zigzag(M[u], M, diff);
      gel(resC,i) = gel(c,1);
      gel(resE,i) = zv_z_mul(gel(c,2), w);
    }
  }
  resC = mergeres(resC, resE);
  return resC;
}

GEN
KzMs_simplify(GEN C)
{
  pari_sp av = avma;
  GEN dims, diffs, M, dims2, diffs2, D, Mk, diffk, Mprec, newind, newindprec, Z;
  long k, u, u2, i;
  dims = gel(C,1);
  diffs = gcopy(gel(C,2)); /* TODO less deep copy? */

  M = steepness_matching(dims,diffs);
  M = gel(M,1);

  /*TODO compute the homotopy inverses f and g*/
  dims2 = zero_zv(lg(dims)-1);
  diffs2 = cgetg(lg(diffs), t_VEC);
  for (k = 1; k < lg(dims2); k++)
  {
    Mk = gel(M,k);
    diffk = gel(diffs,k-1);
    newind = zero_zv(dims[k]);
    for (u = 1; u <= dims[k]; u++) if (!Mk[u])
    {
      dims2[k]++;
      newind[u] = dims2[k];
    }
    if (k > 1)
    {
      D = cgetg(dims2[k]+1,t_VEC);
      u2 = 0;
      for (u = 1; u <= dims[k]; u++)
        if (!Mk[u])
        {
          u2++;
          Z = zigzag(u,Mprec,diffk);
          C = gel(Z,1);
          for (i = 1; i < lg(C); i++) C[i] = newindprec[C[i]];
          gel(D,u2) = Z;
        }
      gel(diffs2,k-1) = D;
    }
    Mprec = Mk;
    newindprec = newind;
  }

  return gerepilecopy(av,mkvec2(dims2,diffs2));
}

